<?php

namespace GeoSot\EnvEditor\Tests\Unit\Helpers;

use GeoSot\EnvEditor\EnvEditor;
use GeoSot\EnvEditor\Exceptions\EnvException;
use GeoSot\EnvEditor\Helpers\EnvKeysManager;
use GeoSot\EnvEditor\Tests\TestCase;
use Illuminate\Config\Repository;
use Illuminate\Filesystem\Filesystem;

class EnvKeysManagerTest extends TestCase
{
    protected function setUp(): void
    {
        parent::setUp(); // TODO: Change the autogenerated stub
        $this->app['config']->set('env-editor.paths.env', self::getTestPath());
        $this->app['config']->set('env-editor.envFileName', self::getTestFile());
    }

    /**
     * @test
     */
    public function check_key_existence(): void
    {
        self::assertTrue($this->getEnvKeysManager()->keyExists('LOG_CHANNEL'));
        self::assertTrue($this->getEnvKeysManager()->keyExists('DB_CONNECTION'));
        self::assertFalse($this->getEnvKeysManager()->keyExists('FOO'));
        self::assertFalse($this->getEnvKeysManager()->keyExists('null'));
    }

    /**
     * @test
     */
    public function returns_value_or_default(): void
    {
        self::assertEquals('stack', $this->getEnvKeysManager()->getKey('LOG_CHANNEL'));
        self::assertEquals('mysql', $this->getEnvKeysManager()->getKey('DB_CONNECTION'));
        self::assertEquals('3306', $this->getEnvKeysManager()->getKey('DB_PORT'));
        self::assertEquals('', $this->getEnvKeysManager()->getKey('BROADCAST_DRIVER'));
        self::assertEquals('foo', $this->getEnvKeysManager()->getKey('BROADCAST_DRIVER', 'foo'));
        self::assertEquals(null, $this->getEnvKeysManager()->getKey('FOO'));
        self::assertEquals('Bar', $this->getEnvKeysManager()->getKey('FOO', 'Bar'));
    }

    /**
     * @test
     */
    public function deletes_keys(): void
    {
        $fileName = 'dummy.tmp';
        $fullPath = $this->createNewDummyFile($fileName);
        $this->app['config']->set('env-editor.envFileName', $fileName);

        self::assertStringContainsString('LOG_CHANNEL', file_get_contents($fullPath));
        self::assertTrue($this->getEnvKeysManager()->deleteKey('LOG_CHANNEL'));
        self::assertStringNotContainsString('LOG_CHANNEL=stack', file_get_contents($fullPath));

        self::assertStringContainsString('CACHE_DRIVER', file_get_contents($fullPath));
        self::assertTrue($this->getEnvKeysManager()->deleteKey('CACHE_DRIVER'));
        self::assertStringNotContainsString('CACHE_DRIVER="file"', file_get_contents($fullPath));

        self::assertStringNotContainsString('CACHE_DRIVER', file_get_contents($fullPath));
        try {
            $this->getEnvKeysManager()->deleteKey('CACHE_DRIVER');
        } catch (\Exception $e) {
            self::assertInstanceOf(EnvException::class, $e);
            unlink($fullPath);
        }
    }

    /**
     * @test
     */
    public function edits_keys(): void
    {
        $fileName = 'dummy.tmp';
        $fullPath = $this->createNewDummyFile($fileName);
        $this->app['config']->set('env-editor.envFileName', $fileName);

        self::assertStringContainsString('LOG_CHANNEL=stack', file_get_contents($fullPath));
        self::assertTrue($this->getEnvKeysManager()->editKey('LOG_CHANNEL', 'foo'));
        self::assertStringContainsString('LOG_CHANNEL=foo', file_get_contents($fullPath));

        self::assertStringContainsString('CACHE_DRIVER="file"', file_get_contents($fullPath));
        self::assertTrue($this->getEnvKeysManager()->editKey('CACHE_DRIVER', '"bar"'));
        self::assertStringContainsString('CACHE_DRIVER="bar"', file_get_contents($fullPath));

        self::assertTrue($this->getEnvKeysManager()->editKey('CACHE_DRIVER', ''));
        self::assertStringContainsString('CACHE_DRIVER=', file_get_contents($fullPath));

        self::assertTrue($this->getEnvKeysManager()->editKey('CACHE_DRIVER', null));
        self::assertStringContainsString('CACHE_DRIVER=', file_get_contents($fullPath));

        self::assertStringNotContainsString('WRONG_KEY', file_get_contents($fullPath));
        try {
            $this->getEnvKeysManager()->editKey('WRONG_KEY', 'fail');
        } catch (\Exception $e) {
            self::assertInstanceOf(EnvException::class, $e);
            unlink($fullPath);
        }
    }

    /**
     * @param  array<string, mixed>  $config
     * @return EnvKeysManager
     */
    protected function getEnvKeysManager(array $config = []): EnvKeysManager
    {
        $envEditor = new EnvEditor(
            new Repository($config ?: $this->app['config']->get('env-editor')),
            new Filesystem()
        );
        $this->app->singleton(EnvEditor::class, function () use ($envEditor) {
            return $envEditor;
        });

        return $envEditor->getKeysManager();
    }

    protected function createNewDummyFile(string $name = 'test.tmp'): string
    {
        $dummyFullPath = self::getTestPath().DIRECTORY_SEPARATOR.$name;

        copy(self::getTestFile(true), $dummyFullPath);

        return $dummyFullPath;
    }
}
